/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

const float RAINBOW_SPLINE_SIZE = 6.0;

/**
 * Lookup table for rainbox colors. Workaround as GLSL does not support arrays.
 * @param i: Array index; Should be integer.
 * @return The color at the given index.
 */
vec3 GetRainbowColor(float i){
	if(i == 0.0){
		return vec3(1.0, 0.0, 0.0); // Red
	}
    else if(i == 1.0){
		return vec3(1.0, 0.5, 0.0); // Orange
	}
	else if(i == 2.0){
		return vec3(1.0, 1.0, 0.0); // Yellow
	}
	else if(i == 3.0){
		return vec3(0.0, 1.0, 0.0); // Green	
    }
	else if(i == 4.0){
		return vec3(0.0, 0.0, 1.0); // Blue	
	}
	else if (i == 5.0){
        return vec3(0.27, 0.0, 0.51); // Purple
	}
	else if (i == 6.0){
		return vec3(0.57, 0.0, 1.0); // Violet	
	}

	return vec3(1.0, 1.0, 1.0); // should never get here.
}

/**
 * Perform Catmull-Rom spline interpolation between support points v1 and v2.
 * @param x: Interpolation factor between v1 and v2; Range[0.0,1.0]
 * @param v0: left most control point.
 * @param v1: left support point.
 * @param v2: right support point.
 * @param v3: right most control point.
 * @return The interpolated value.
 */
vec3 CatmullRomSpline(float x, vec3 v0, vec3 v1, vec3 v2, vec3 v3) 
{
    // Note: this spline will go through it's support points.
	vec3 c2 = -.5 * v0				+ 0.5 *v2;
	vec3 c3 = 		v0	+ -2.5*v1 	+ 2.0 *v2 + -.5*v3;
	vec3 c4 = -.5 * v0	+ 1.5 *v1 	+ -1.5*v2 + 0.5*v3;
	return(((c4 * x + c3) * x + c2) * x + v1);
}

/**
 * Evaluates the rainbox texture in UV-space using a Catmull-Rom spline.
 */
vec3 EvaluateRainbowSpline(float x){
    // x must be in range [0.0,1.0]
    float scaledX = clamp(x, 0.0, 1.0) * RAINBOW_SPLINE_SIZE;
    
    // Determine which 'rainbox segment' we are evluating:
    float segmentIndex = floor(scaledX);
    
    // Note that you evaluate between v1 and v2, using v0 and v3 as control points:
    vec3 v0 = GetRainbowColor(segmentIndex-1.0);
    vec3 v1 = GetRainbowColor(segmentIndex+0.0);
    vec3 v2 = GetRainbowColor(segmentIndex+1.0);
    vec3 v3 = GetRainbowColor(segmentIndex+2.0);
    
    return CatmullRomSpline(fract(scaledX), v0,v1,v2,v3);
}

void main(void)
{
	vec2 uv = gl_FragCoord.xy / iResolution.xy; // in range [0.0,1.0]

    gl_FragColor = vec4(EvaluateRainbowSpline(uv.x),1.0);
}